<?php
namespace model;

use \lib\Session as Session;

class BaseUser extends \Pawn\Model
{
	// Migration
	public static function migration($model_name)
	{
		if($model_name === 'BaseUser')
		{
			return false;
		}
		
		return array
		(
			array('CREATE', array
			(
				'username'		=>	'VARCHAR(16) UNIQUE',
				'password'		=>	'CHAR(32)',
				'new_password'	=>	'CHAR(32) DEFAULT ""',
				'email'			=>	'VARCHAR(255)',
				'status'		=>	'INT UNSIGNED',
				'level'			=>	'TINYINT'
			))
		);
	}
	
	// Identifier
	public function __toString()
	{
		return $this->username;
	}
	
	// Administration
	const sortColumn = 'ts_edit';
	const sortAscending = 0;
	
	public function listing()
	{
		return array
		(
			'username'		=>	$this->__toString(),
			'level'			=>	\lib\Lang::get('user_level', $this->level),
			'email'			=>	$this->email,
			'ts_create'		=>	\Pawn::get_date($this->ts_create),
			'ts_edit'		=>	\Pawn::get_date($this->ts_edit)
		);
	}
	
	public static function actions()
	{
		return array('add');
	}
	
	public static function bulk()
	{
		return array
		(
			'trash'		=>	true,
			'edit'		=>	false
		);
	}
	
	// Validation
	public function columns($preset, $action, $form)
	{
		if($action == 'create' || $preset == 'administrate')
		{
			$columns = array
			(
				'username'	=>	array('text', array('min_length'=>3, 'max_length'=>16, 'alphanumeric'=>true, 'c_unique_username'=>array($this, 'unique_username'))),
			);
		} else
		{
			$columns = array();
		}
		
		$columns = array_merge($columns, array
		(
			'password'			=>	array('password', array('min_length'=>5)),
			'password_repeat'	=>	array('password', array('match'=>'password')),
			'email'				=>	array('text', array('max_length'=>255, 'email'=>true))
		));
		
		if($action == 'edit')
		{
			$columns['password'][1]['optional'] = true;
			
			if($form->request('password') == '')
			{
				$columns['password_repeat'][1]['optional'] = true;
			}
			
			if($preset == 'profile')
			{
				$columns['current_password'] = array('password', array('c_current_password'=>array($this, 'current_password')));
			}
		}
		
		if($preset == 'administrate')
		{
			$levels = array(0=>1, 2=>1, 3=>1, 4=>1, 5=>1);
			
			foreach($levels as $level=>$value)
			{
				$levels[$level] = \lib\Lang::get('user_level', $level);
			}
			
			$columns['level'] = array('select', $levels, 2, array
			(
				'c_level'	=>	array($this, 'validate_level'),
				'c_demote'	=>	array($this, 'demote'),
				'c_promote'	=>	array($this, 'promote')
			), false);
		}
		
		return $columns;
	}
	
	public function unique_username($value)
	{
		if($this->id)
		{
			return self::get('WHERE id!=? AND username=?', array($this->id, $value), false)->fetch() === false;
		} else
		{
			return self::get('WHERE username=?', $value, false)->fetch() === false;
		}
	}
	
	public function validate_level($value)
	{
		global $autoload;
		
		return $this->id != $autoload['Session']['user']->level || $this->level == $value;
	}
	
	public function demote($value)
	{
		global $autoload;
		
		return $this->level == $value || $this->level < $autoload['Session']['user']->level;
	}
	
	public function promote($value)
	{
		global $autoload;
		
		return $this->level == $value || $value < $autoload['Session']['user']->level;
	}
	
	public function fields($action, $fields, $preset)
	{
		$row = $this->allow_fields($fields, array('username', 'email', 'level'));
		
		if($row['username'] === '')
		{
			unset($row['username']);
		}
		
		if($row['level'] === '')
		{
			unset($row['level']);
		}
		
		$row['status'] = time();
		
		if(isset($fields['password']) && $fields['password'] != '')
		{
			$row['password'] = hash_hmac('md5', $fields['password'], SECRET);
		}
		
		if($action == 'create')
		{
			if($preset == 'super')
			{
				$row['level'] = 5;
			} elseif($preset == 'register')
			{
				$row['level'] = 2;
			}
		}
		
		return $row;
	}
	
	// Authentication
	public static function authenticate($name, $password=false, $remember=false)
	{
		if($password !== false)
		{
			$user = self::get('WHERE username=? AND password=? AND level>?', array($name, hash_hmac('md5', $password, SECRET), 1))->fetch();
		} else
		{
			$user = self::get('WHERE id=?', $name)->fetch();
		}
		
		if($user)
		{
			Session::set('user', $user->id.' '.$user->password, $remember?strtotime('+10 years'):0);
		}
		
		return $user;
	}
	
	public static function unauthenticate()
	{
		Session::delete('user');
		
		return self::anonymous();
	}
	
	// Session
	public static function session()
	{
		if($data = Session::get('user'))
		{
			$data = explode(' ', $data);
			
			if($user = self::get('WHERE id=? AND level>? AND password=?', array($data[0], 1, $data[1]))->fetch())
			{
				$user->status = time();
				
				return $user;
			}
		}
		
		return self::anonymous();
	}
	
	// Anonymous
	private static function anonymous()
	{
		return new User(array
		(
			'id'	=>	0,
			'level'	=>	1,
			'username'	=>	\lib\Lang::get('anonymous')
		));
	}
	
	// Authenticated
	public function authenticated()
	{
		return $this->level != 1;
	}
	
	// Trash
	public function trash()
	{
		$user = Session::user();
		
		if($user->id != $this->id && $user->level > $this->level)
		{
			parent::trash();
		}
	}
	
	// Log in
	public static function log_in_form()
	{
		$form = new \lib\Form('log_in_form');
			
		$form->field('username', 'text', array
		(
			'c_user_exists'	=>	function($username)
			{
				return \model\User::num_rows('WHERE username=?', $username, false);
			},
			'c_user_banned'	=>	function($username)
			{
				return \model\User::num_rows('WHERE username=? AND level>1', $username);
			}
		));
		
		$form->field('password', 'password', array
		(
			'c_user_password'	=>	function($password, $null, $form)
			{
				return \model\User::num_rows('WHERE username=? AND password=?', array($form->request('username'), hash_hmac('md5', $password, SECRET)));
			}
		));
		
		$form->field('remember', 'checkbox');
		
		if($form->validate())
		{
			self::authenticate($form->username->value, $form->password->value, $form->remember->value);
		}
		
		return $form;
	}
	
	// Register
	public static function register_form($enter=true)
	{
		$user = new \model\User;
		
		$form = $user->create('register_form', 'register');
		
		if($enter && $form->valid)
		{
			self::authenticate($form->username->value, $form->password->value);
		}
		
		return $form;
	}
	
	// Edit profile
	
	public function current_password($password)
	{
		return hash_hmac('md5', $password, SECRET) == $this->password;
	}
	
	// Forgot password
	public static function forgot_password_form()
	{
		if(isset($_GET['fp']) && isset($_GET['id']))
		{
			\lib\db::query('UPDATE '.self::table_name().' SET password=new_password WHERE id=? AND new_password=?', array($_GET['id'], hash_hmac('md5', \lib\Session::decrypt($_GET['fp']), SECRET)));
		}
		
		$form = new \lib\Form('forgot_password_form');
		
		$form->field('username', 'text', array
		(
			'c_user_exists'	=>	function($username)
			{
				return \model\User::num_rows('WHERE username=?', $username, false);
			}
		));
		
		$form->field('email', 'text', array
		(
			'email'			=>	true,
			'c_user_email'	=>	function($email, $null, $form)
			{
				return \model\User::num_rows('WHERE username=? AND email=?', array($form->request('username'), $email));
			}
		));
		
		if($form->validate())
		{
			$user = \model\User::get('WHERE username=?', $form->request('username'))->fetch();
			
			$password = uniqid();
			
			$user->new_password = hash_hmac('md5', $password, SECRET);
			
			$lang = \lib\Lang::get('request_password_mail');
			
			$message = $lang[1].":\n".$password."\n\n".$lang[2].":\nhttp".(isset($_SERVER['HTTPS'])?'s':'')."://".$_SERVER['HTTP_HOST'].WEB."?fp=".\lib\Session::encrypt($password)."&id=".$user->id."\n\n".$lang[3];
			
			\Pawn::send_mail($user->email, $lang[0], $message);
		}
		
		return $form;
	}
	
	// Get
	public static function get($query=null, $data=array(), $trash=0, $call=null)
	{
		if($query == 'WHERE id=?' && $data == '0')
		{
			return self::anonymous();
		}
		
		return parent::get($query, $data, $trash, $call);
	}
	
	// Fetch
	public function fetch()
	{
		return $this;
	}
	
	// Profile
	public function profile()
	{
		if($this->authenticated())
		{
			return '<a class="user-level-'.$this->level.'" href="'.\Pawn::path('Users', 'profile', array('name'=>$this->username)).'">'.$this->username.'</a>';
		} else
		{
			return '<span class="user-level-1">'.\lib\Lang::get('anonymous').'</span>';
		}
	}
}